const UserModel = require('../models/userModel');
const EncryptionUtil = require('../utils/encryption');
const generateOTP = require('../utils/otpGenerator');

class AuthController {

    // المرحلة 1: التحقق من الباسورد وإرسال OTP
    static async login(req, res) {
        try {
            const { email, password } = req.body;

            // البحث عن المستخدم
            const user = await UserModel.findByEmail(email);
            if (!user) {
                return res.status(401).json({ message: 'بيانات الدخول غير صحيحة' });
            }

            // مطابقة كلمة المرور
            const isMatch = await EncryptionUtil.comparePassword(password, user.password_hash);
            if (!isMatch) {
                return res.status(401).json({ message: 'بيانات الدخول غير صحيحة' });
            }

            // إنشاء OTP وصلاحيته (5 دقائق)
            const otp = generateOTP();
            const expiry = new Date(Date.now() + 5 * 60 * 1000);

            // حفظ الرمز في قاعدة البيانات
            await UserModel.saveOtp(user.id, otp, expiry);

            // طباعة الرمز في الكونسول (لأغراض التطوير)
            console.log(`🔐 رمز التحقق للمستخدم ${email}: ${otp}`);

            res.json({ 
                message: 'تم التحقق. أدخل رمز OTP المرسل إليك.', 
                userId: user.id 
            });

        } catch (error) {
            console.error(error);
            res.status(500).json({ message: 'خطأ في السيرفر' });
        }
    }

    // المرحلة 2: التحقق من OTP وإصدار التوكن
    static async verifyOTP(req, res) {
        try {
            const { userId, otpCode } = req.body;
            const user = await UserModel.findById(userId);

            if (!user || user.otp_code !== otpCode) {
                return res.status(400).json({ message: 'رمز التحقق غير صحيح' });
            }

            if (new Date() > user.otp_expiry) {
                return res.status(400).json({ message: 'انتهت صلاحية الرمز' });
            }

            // تنظيف الرمز
            await UserModel.clearOtp(userId);

            // إنشاء التوكن
            const token = EncryptionUtil.generateToken(user);

            res.json({ 
                message: 'تم تسجيل الدخول بنجاح', 
                token, 
                user: { username: user.username, role: user.role } 
            });

        } catch (error) {
            console.error(error);
            res.status(500).json({ message: 'خطأ أثناء التحقق' });
        }
    }
}

module.exports = AuthController;