const AssetModel = require('../models/assetModel');
const LogModel = require('../models/logModel'); // 👈 استيراد المودل الجديد

class AssetController {
    
    static async getAll(req, res) {
        try {
            const assets = await AssetModel.getAllAssets();
            res.json(assets);
        } catch (error) {
            res.status(500).json({ message: error.message });
        }
    }

    static async create(req, res) {
        try {
            const assetData = { ...req.body, createdBy: req.user.id };
            const newAssetId = await AssetModel.createAsset(assetData);

            // 📝 تسجيل النشاط
            await LogModel.createLog(
                req.user.id, 
                req.user.username || 'مستخدم', 
                'إضافة', 
                newAssetId, 
                `تم إضافة أصل جديد: ${assetData.name}`
            );

            res.status(201).json({ message: 'تم الإضافة', id: newAssetId });
        } catch (error) {
            res.status(500).json({ message: error.message });
        }
    }

    static async update(req, res) {
        try {
            const id = req.params.id;
            const updatedData = req.body;
            
            const success = await AssetModel.updateAsset(id, updatedData);
            
            if (success) {
                // 📝 تسجيل النشاط
                await LogModel.createLog(
                    req.user.id, 
                    req.user.username || 'مستخدم', 
                    'تعديل', 
                    id, 
                    `تم تعديل بيانات الأصل: ${updatedData.name}`
                );
                res.json({ message: 'تم التحديث' });
            } else {
                res.status(404).json({ message: 'الأصل غير موجود' });
            }
        } catch (error) {
            res.status(500).json({ message: error.message });
        }
    }

    static async delete(req, res) {
        try {
            const id = req.params.id;
            // نجلب بيانات الأصل قبل الحذف لتسجيل اسمه
            const asset = await AssetModel.getAssetById(id);
            
            const success = await AssetModel.deleteAsset(id);
            
            if (success) {
                // 📝 تسجيل النشاط
                await LogModel.createLog(
                    req.user.id, 
                    req.user.username || 'مستخدم', 
                    'حذف', 
                    id, 
                    `تم حذف الأصل: ${asset ? asset.name : 'غير معروف'}`
                );
                res.json({ message: 'تم الحذف' });
            } else {
                res.status(404).json({ message: 'الأصل غير موجود' });
            }
        } catch (error) {
            res.status(500).json({ message: error.message });
        }
    }
// استيراد مجمع
    static async importBulk(req, res) {
        try {
            const assets = req.body; // مصفوفة من الأصول
            
            if (!Array.isArray(assets) || assets.length === 0) {
                return res.status(400).json({ message: 'لا توجد بيانات صالحة للاستيراد' });
            }

            console.log(`📦 بدء استيراد ${assets.length} أصل...`);
            
            const count = await AssetModel.createBulk(assets, req.user.id);

            // تسجيل النشاط
            const LogModel = require('../models/logModel');
            await LogModel.createLog(
                req.user.id, 
                req.user.username, 
                'إضافة', 
                0, 
                `تم استيراد ${count} أصل عبر ملف CSV`
            );

            res.status(201).json({ message: `تم استيراد ${count} أصل بنجاح! ✅` });

        } catch (error) {
            // التعامل مع خطأ تكرار الكود (Duplicate Entry)
            if (error.code === 'ER_DUP_ENTRY') {
                return res.status(400).json({ message: 'يوجد رموز أصول (Codes) مكررة في الملف أو موجودة مسبقاً.' });
            }
            res.status(500).json({ message: error.message });
        }
    }
}

module.exports = AssetController;