const db = require('../config/db');

class AnalysisController {

    // 1. فحص الثغرات
    static async scanVulnerabilities(req, res) {
        try {
            console.log('🛡️ بدء فحص الثغرات...');
            const vulnerabilities = [];

            // قاعدة 1: فحص الأنظمة القديمة (End of Life)
            const [oldOsAssets] = await db.query(`
                SELECT id, name, code, operating_system 
                FROM assets 
                WHERE operating_system LIKE '%Windows 7%' 
                   OR operating_system LIKE '%Windows 8%' 
                   OR operating_system LIKE '%Server 2008%'
            `);

            oldOsAssets.forEach(asset => {
                vulnerabilities.push({
                    assetId: asset.id,
                    assetName: asset.name,
                    severity: 'عالية جداً',
                    issue: 'نظام تشغيل منتهي الصلاحية (EOL)',
                    description: `الجهاز يعمل بنظام ${asset.operating_system} غير المدعوم أمنياً.`
                });
            });

            // قاعدة 2: ضمان منتهي (مخاطرة تشغيلية)
            const [expiredWarranty] = await db.query(`
                SELECT id, name, code, warranty_expiry 
                FROM assets 
                WHERE warranty_expiry < NOW()
            `);

            expiredWarranty.forEach(asset => {
                vulnerabilities.push({
                    assetId: asset.id,
                    assetName: asset.name,
                    severity: 'متوسطة',
                    issue: 'ضمان منتهي',
                    description: 'انتهى ضمان الجهاز، مما يشكل خطراً في حال تعطل الهاردوير.'
                });
            });

            // قاعدة 3: أجهزة بدون تحديث (لم يراها الايجنت منذ شهر)
            const [lostAgents] = await db.query(`
                SELECT id, name, last_discovered 
                FROM assets 
                WHERE discovery_method = 'Agent' 
                  AND last_discovered < DATE_SUB(NOW(), INTERVAL 30 DAY)
            `);

            lostAgents.forEach(asset => {
                vulnerabilities.push({
                    assetId: asset.id,
                    assetName: asset.name,
                    severity: 'عالية',
                    issue: 'انقطاع الاتصال',
                    description: 'لم يتم استلام تحديثات من الايجنت منذ أكثر من 30 يوماً.'
                });
            });

            res.json({
                count: vulnerabilities.length,
                details: vulnerabilities
            });

        } catch (error) {
            console.error(error);
            res.status(500).json({ message: 'فشل الفحص' });
        }
    }

    // 2. تقرير الامتثال
    static async checkCompliance(req, res) {
        try {
            console.log('📊 حساب نسبة الامتثال...');
            
            // إحضار كل الأصول
            const [assets] = await db.query('SELECT * FROM assets');
            const totalAssets = assets.length;
            
            if (totalAssets === 0) return res.json({ score: 100, issues: [] });

            let issues = [];
            let compliantCount = 0;

            assets.forEach(asset => {
                let assetIssues = [];

                // قاعدة 1: هل للأصل مسؤول؟
                if (!asset.owner || asset.owner === 'غير محدد') assetIssues.push('لا يوجد مسؤول (Owner)');
                
                // قاعدة 2: هل الموقع محدد؟
                if (!asset.location) assetIssues.push('الموقع غير محدد');

                // قاعدة 3: هل المعلومات المالية مكتملة؟
                if (!asset.value || asset.value == 0) assetIssues.push('القيمة المالية مفقودة');

                if (assetIssues.length === 0) {
                    compliantCount++;
                } else {
                    issues.push({
                        code: asset.code,
                        name: asset.name,
                        missing: assetIssues
                    });
                }
            });

            const score = Math.round((compliantCount / totalAssets) * 100);

            res.json({
                score: score,
                total: totalAssets,
                compliant: compliantCount,
                nonCompliant: issues
            });

        } catch (error) {
            console.error(error);
            res.status(500).json({ message: 'فشل التقرير' });
        }
    }
}

module.exports = AnalysisController;