const db = require('../config/db');
const os = require('os'); // مكتبة مدمجة في Node.js لمعرفة حالة الجهاز

class SystemController {

    // 1. فحص حالة النظام الحقيقي (Health Check)
    static async getHealth(req, res) {
        const start = Date.now();
        try {
            // فحص قاعدة البيانات
            await db.query('SELECT 1');
            const dbLatency = Date.now() - start;

            // معلومات السيرفر
            const memoryUsage = process.memoryUsage().heapUsed / 1024 / 1024; // MB
            const uptime = process.uptime(); // Seconds
            
            res.json({
                status: 'healthy',
                db_status: 'Connected',
                db_latency: `${dbLatency}ms`,
                server_uptime: `${(uptime / 60).toFixed(2)} Minutes`,
                memory_usage: `${memoryUsage.toFixed(2)} MB`,
                os_load: os.loadavg()[0].toFixed(2)
            });

        } catch (error) {
            res.status(500).json({
                status: 'unhealthy',
                db_status: 'Disconnected',
                error: error.message
            });
        }
    }

    // 2. جلب إعدادات التكامل
    static async getSettings(req, res) {
        try {
            const [rows] = await db.query('SELECT * FROM system_settings');
            const settings = {};
            rows.forEach(row => {
                settings[row.setting_key] = row.setting_value;
            });
            res.json(settings);
        } catch (error) {
            res.status(500).json({ message: error.message });
        }
    }

    // 3. حفظ إعدادات التكامل
    static async saveSettings(req, res) {
        try {
            const settings = req.body; // { ad_server: '...', ad_domain: '...' }
            
            // تحديث القيم واحدة تلو الأخرى
            for (const [key, value] of Object.entries(settings)) {
                await db.query(
                    'INSERT INTO system_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?',
                    [key, value, value]
                );
            }

            // تسجيل النشاط
            const LogModel = require('../models/logModel');
            await LogModel.createLog(req.user.id, req.user.username, 'تعديل', 0, 'تحديث إعدادات تكامل النظام');

            res.json({ message: 'تم حفظ الإعدادات بنجاح ✅' });

        } catch (error) {
            res.status(500).json({ message: error.message });
        }
    }
}

module.exports = SystemController;